#!/bin/bash
# set -x
###############################################################################
#
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corp. 2021, 2022. All Rights Reserved.
#
# US Government Users Restricted Rights - Use, duplication or
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
#
###############################################################################
CUR_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
PARENT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )/.." && pwd )"
CLI_CMD=kubectl

# Import common utilities and environment variables
source ${CUR_DIR}/helper/common.sh

SCRIPT_NAME='cp4a-edb-production-backup-restore.sh'
CP4BA_OPERATOR="ibm-cp4a-operator"
CONTENT_OPERATOR="ibm-content-operator"
EDB="EnterpriseDB Postgres for Kubernetes"
CP4BA="Cloud Pak for Business Automation"
# AWS Access Key
ACCESS_KEY_ID=""
ACCESS_SECRET_KEY=""
ACCESS_SESSION_TOKEN=""
TMP_EDB_DIR="${CUR_DIR}/EDB"
S3_SECRET_YAML_FILE="${TMP_EDB_DIR}/aws-creds.yaml"
AZ_SECRET_YAML_FILE="${TMP_EDB_DIR}/azure-creds.yaml"
GOOGLE_SECRET_SCRIPT_FILE="${TMP_EDB_DIR}/create-google-secret.sh"
S3_SECRET_NAME="aws-creds"
AZ_SECRET_NAME="azure-creds"
GOOGLE_SECRET_NAME="backup-creds"

# EDB default values
EDB_CLUSTER_NAME="postgres-cp4ba"
EDB_CLUSTER_YAML_FILE="${TMP_EDB_DIR}/${EDB_CLUSTER_NAME}.yaml"
EDB_CLUSTER_YAML_FILE_OG="${TMP_EDB_DIR}/${EDB_CLUSTER_NAME}-original.yaml"
EDB_CLUSTER_SCHEDULE_BACKUP="${TMP_EDB_DIR}/${EDB_CLUSTER_NAME}-schedule-backup.yaml"

EDB_RECOVERY_NAME="clusterBackup"
# Next Steps messages.
# 1 create secret 
# 2 apply the cluster cr
# 3 apply the schedule back
NEXT_STEP="To start backups for your $EDB deployment for $CP4BA, follow these next steps.\n"
NEXT_STEP_0="0. Please verify the generate secret and cluster yaml files found in ${TMP_EDB_DIR}.\n"
NEXT_STEP_1="1. Please create the secret with your credentials "
NEXT_STEP_2="2. Please apply the Cluster CR by running \n\t- ${CLI_CMD} apply -f ${EDB_CLUSTER_YAML_FILE}\n"
NEXT_STEP_3="3. Once you're ready apply the ScheduledBackup to start backing up your database.\n\t- ${CLI_CMD} apply -f ${EDB_CLUSTER_SCHEDULE_BACKUP}\n"


NEXT_STEP_RECOVERY="To start recovery for your $EBD deployment for $CP4BA, please follow the following steps.\n"
NEXT_STEP_RECOVERY_0="${NEXT_STEP_0}"
NEXT_STEP_RECOVERY_1="${NEXT_STEP_1}"
NEXT_STEP_RECOVERY_2="2. Please scale down the operators \n\t- ${CLI_CMD} scale deployment ${CP4BA_OPERATOR} ${CONTENT_OPERATOR} --replicas=0\n"
NEXT_STEP_RECOVERY_3="3. Please scale down the deployments that are leveraging the $EDB deployment for $CP4BA.\n\t Components that support $EDB  \n\t\t- Content \n\t\t- Automation Document Processing \n\t\t- Business Automation Workflow\n\t\t- Business Automation Studio\n\t\t- Operational Decision Manager\n"
NEXT_STEP_RECOVERY_4="4. Once all the necessary deployments are scaled down, we want to delete the $EDB deployment '$EDB_CLUSTER_NAME'.\n\t- ${CLI_CMD} delete -f ${EDB_CLUSTER_YAML_FILE}\n"
NEXT_STEP_RECOVERY_5="5. Please re-create the $EDB deployment '$EDB_CLUSTER_NAME' with the yaml file ${EDB_CLUSTER_YAML_FILE}.\n\t- ${CLI_CMD} apply -f ${EDB_CLUSTER_YAML_FILE}\n"
NEXT_STEP_RECOVERY_6="6. Wait until the recovery process has completed. When the pod '${EDB_CLUSTER_NAME}-1' is in a running state, the recovery process has completed.\n"
NEXT_STEP_RECOVERY_7="7. Please scale up the operator deployments.\n\tImportant: Since we had to recreate the $EDB deployment '$EDB_CLUSTER_NAME', there's new certificates that was generated by the postgres operator.\n\tThe $CP4BA operator will need to regenerate a new client certificate for the applications to use.\n\tIt could take two reconciles for the ${CONTENT_OPERATOR} operator to update the FNCM deployments with the latest client certificate.\n\n\t- ${CLI_CMD} scale deployment ${CP4BA_OPERATOR} ${CONTENT_OPERATOR} --replicas=1\n"
NEXT_STEP_RECOVERY_8=""


mkdir -p ${TMP_EDB_DIR}

function select_project(){

    while [[ $TARGET_PROJECT_NAME == "" ]];
    do
        if [ -z "$CP4BA_AUTO_NAMESPACE" ]; then
            echo
            # echo -e "\x1B[1mWhere do you want to deploy Cloud Pak for Business Automation?\x1B[0m"
            read -p "Enter the name for an existing project (namespace): " TARGET_PROJECT_NAME
        else
            if [[ "$CP4BA_AUTO_NAMESPACE" == openshift* ]]; then
                echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'openshift' or start with 'openshift' \x1B[0m"
                exit 1
            elif [[ "$CP4BA_AUTO_NAMESPACE" == kube* ]]; then
                echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'kube' or start with 'kube' \x1B[0m"
                exit 1
            fi
            TARGET_PROJECT_NAME=$CP4BA_AUTO_NAMESPACE
        fi

        if [ -z "$TARGET_PROJECT_NAME" ]; then
            echo -e "\x1B[1;31mEnter a valid project name, project name can not be blank\x1B[0m"
        elif [[ "$TARGET_PROJECT_NAME" == openshift* ]]; then
            echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'openshift' or start with 'openshift' \x1B[0m"
            TARGET_PROJECT_NAME=""
        elif [[ "$TARGET_PROJECT_NAME" == kube* ]]; then
            echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'kube' or start with 'kube' \x1B[0m"
            TARGET_PROJECT_NAME=""
        else
            isProjExists=`${CLI_CMD} get project $TARGET_PROJECT_NAME --ignore-not-found | wc -l`  >/dev/null 2>&1

            if [ "$isProjExists" -ne 2 ] ; then
                echo -e "\x1B[1;31mInvalid project name, please enter a existing project name ...\x1B[0m"
                TARGET_PROJECT_NAME=""
            else
                echo -e "\x1B[1mUsing project ${TARGET_PROJECT_NAME}...\x1B[0m"
            fi
        fi
    done
  return 0
}

function display_msgs() {
  var=("$@")
  for v in "${var[@]}";
  do
    printf "$v"
  done
}


function enter_input() {

  local req=""
  local prompt=$1
  local opt=$2

  while read -e $opt -p "${req}${prompt}" prompt_ans
  do
    #echo "${req}${prompt}"
    #read -r  $opt  prompt_ans
    if [[ "$prompt_ans" != "" ]] ; then
      req=""
      echo -n "${prompt_ans}"
      break;
    else
      req="$RED_TEXT[REQUIRED]$RESET_TEXT "

    fi
  done
}

function display_input_custom() {

  local req=""
  local prompt=$1
  local arr=("${@:2}")
   
  while read -e  -p "${req}${prompt}" prompt_ans
  do
    if [[ "${prompt_ans}" != "" ]] ; then
      case "${arr[@]}" in
      *"${prompt_ans}"* ) 
        echo -n "${prompt_ans}"
        break
        ;;
      * )
        req="$RED_TEXT[REQUIRED]$RESET_TEXT "
        ;;
      esac
    else 
      req="$RED_TEXT[REQUIRED]$RESET_TEXT "
    fi
  done

}

function display_input_yes_no() {

  local req=""
  local prompt_optional=$1

  while read -e -p "${req}${prompt_optional}" prompt_ans
  do
    if [[ "$prompt_ans" != "" && ( "$prompt_ans" == "No"  || "$prompt_ans" == "NO" || "$prompt_ans" == "nO" || "$prompt_ans" == "no"  || "$prompt_ans" == "N"  || "$prompt_ans" == "n"  ) ]] ; then
      req=""
      echo "no"
      break
    elif [[ "$prompt_ans" != "" && ( "$prompt_ans" == "Yes"  || "$prompt_ans" == "yes"  || "$prompt_ans" == "YeS"  || "$prompt_ans" == "yES"  || "$prompt_ans" == "yeS"  || "$prompt_ans" == "Y"  || "$prompt_ans" == "y"  ) ]] ; then
      req=""
      session_token='yes'
      echo $session_token
      #printf "\n"
      break;
    elif [[ "$prompt_ans" == "" ]] ; then
      echo "no"
      break;
    else
      req="$RED_TEXT[REQUIRED]$RESET_TEXT "
      #printf "\n"
    fi
  done

}


function get_edb_cluster() {

  rm -fr ${EDB_CLUSTER_YAML_FILE}
  rm -fr ${EDB_CLUSTER_YAML_FILE_OG}
  #${CLI_CMD} get cluster ${EDB_CLUSTER_NAME} -o yaml > ${EDB_CLUSTER_YAML_FILE}
  ## https://jsw.ibm.com/browse/DBACLD-178382 - Fixing the Error in fetching cluster in postgresedb backup
  ${CLI_CMD} get clusters.postgresql.k8s.enterprisedb.io ${EDB_CLUSTER_NAME} -o yaml > ${EDB_CLUSTER_YAML_FILE}
  cp -f ${EDB_CLUSTER_YAML_FILE} ${EDB_CLUSTER_YAML_FILE_OG}


  ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} status
  ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} spec.backup
  ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} metadata.annotations
  ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} metadata.creationTimestamp
  ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} metadata.generation
  ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} metadata.resourceVersion
  if [[ "${MODE}" == "backup" ]] ; then
    ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} spec.backup
  elif [[  "${MODE}" == "recovery" ]] ; then
    ${YQ_CMD} d -i ${EDB_CLUSTER_YAML_FILE} spec.bootstrap
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].name "${EDB_RECOVERY_NAME}"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.serverName "${EDB_CLUSTER_NAME}"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.bootstrap.recovery.source "${EDB_RECOVERY_NAME}"
  fi
  
}
function aws_s3(){
  if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then
    printf "\n*************** DEBUG MSG ***********************\n"
    printf "Start of function aws_s3() to retrieve user input"
  fi
 

  touch ${S3_SECRET_YAML_FILE}

  local prompt_access_key_id="Enter the ID of the access key that will be used to upload files into s3: "
  local prompt_access_secret_key="Enter the secret part of the access key mentioned above: "
  local prompt_access_session_token="Enter the optional session token, in case it is required: "
  local prompt_optional="Do you have a session token for your access key(Yes/No Default No): " 
  if [[ "${MODE}" == "backup" ]] ; then
    local prompt_destination_path="Enter the destination path. It can by any URL pointing to a folder where the instance can upload the WAL files. ( e.g. s3://BUCKET_NAME/path/to/folder ): " 
  else
    local prompt_destination_path="Enter the destination path. It can by any URL pointing to a folder where the instance can recover from. ( e.g. s3://BUCKET_NAME/path/to/folder ): "
  fi 

  local prompt_other_s3_compatible="Are you using S3-compatible object store, like MinIO or Linode Object Store(Yes/No Default No)"
  local prompt_other_s3_compatible_endpoint="Specify an endpoint instead of using the default S3 one. "
  ACCESS_KEY_ID=$( enter_input "${prompt_access_key_id}" "-s" | base64 -w 0 )
  printf "\n"

  ACCESS_SECRET_KEY=$( enter_input "${prompt_access_secret_key}" "-s" | base64 -w 0 )
  printf "\n"

  # local session_token="no"
  local req=""
 
  local session_token=$( display_input_yes_no "${prompt_optional}" )
  
  if [[ "${session_token}" == "yes" ]] ; then
    ACCESS_SESSION_TOKEN=$( enter_input "$prompt_access_session_token" "-s" | base64 -w 0 )
    printf "\n"
    ${YQ_CMD} w -i $S3_SECRET_YAML_FILE data.ACCESS_SESSION_TOKEN "${ACCESS_SESSION_TOKEN}"
    if [[ "${MODE}" == "backup" ]] ; then
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.s3Credentials.accessSessionToken.name "${S3_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.s3Credentials.accessSessionToken.key "ACCESS_SESSION_TOKEN"
    else 
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.s3Credentials.accessSessionToken.name "${S3_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.s3Credentials.accessSessionToken.key "ACCESS_SESSION_TOKEN"

    fi 
   
  fi

  S3_DESTINATION_PATH=$( enter_input  "$prompt_destination_path" )
  printf "\n"
  # printf $S3_DESTINATION_PATH
  local other_s3_comp=$( display_input_yes_no "${prompt_other_s3_compatible}" )
  if [[ "$other_s3_comp" == "yes" ]] ; then
    END_POINT_URL=$( enter_input "${prompt_other_s3_compatible_endpoint}"  )
  fi 

  ${YQ_CMD} w -i $S3_SECRET_YAML_FILE apiVersion "v1"
  ${YQ_CMD} w -i $S3_SECRET_YAML_FILE kind "Secret"
  ${YQ_CMD} w -i $S3_SECRET_YAML_FILE metadata.name "${S3_SECRET_NAME}"
  ${YQ_CMD} w -i $S3_SECRET_YAML_FILE metadata.namespace "${TARGET_PROJECT_NAME}"
  ${YQ_CMD} w -i $S3_SECRET_YAML_FILE data.ACCESS_SECRET_ID "${ACCESS_KEY_ID}"
  ${YQ_CMD} w -i $S3_SECRET_YAML_FILE data.ACCESS_SECRET_KEY "${ACCESS_SECRET_KEY}"


  if [[ "${MODE}" == "backup" ]] ; then
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.s3Credentials.accessKeyId.name "${S3_SECRET_NAME}"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.s3Credentials.secretAccessKey.name "${S3_SECRET_NAME}"

    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.s3Credentials.accessKeyId.key "ACCESS_SECRET_ID"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.s3Credentials.secretAccessKey.key "ACCESS_SECRET_KEY"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.destinationPath "${S3_DESTINATION_PATH}"
 
    if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then
      printf "\n*************** DEBUG MSG ***********************\n"
      printf "Created file ${EDB_CLUSTER_YAML_FILE} with the updated section for backup. \n"
      cat ${EDB_CLUSTER_YAML_FILE}
      printf "\n"
    fi


    schedule_backups
    printf "\n"

    clear
    local next_steps=( "${NEXT_STEP}" "${NEXT_STEP_0}" "${NEXT_STEP_1} \n\t- ${CLI_CMD} apply -f ${S3_SECRET_YAML_FILE} \n" "${NEXT_STEP_2}" "${NEXT_STEP_3}" )
    display_msgs "${next_steps[@]}"
  else 

    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.s3Credentials.accessKeyId.name "${S3_SECRET_NAME}"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.s3Credentials.secretAccessKey.name "${S3_SECRET_NAME}"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.s3Credentials.accessKeyId.key "ACCESS_SECRET_ID"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.s3Credentials.secretAccessKey.key "ACCESS_SECRET_KEY"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.destinationPath "${S3_DESTINATION_PATH}"
    # EXTERNAL_CLUSTERS=$(  ${YQ_CMD} r  $EDB_CLUSTER_YAML_FILE  -j 'spec.externalClusters' )
    printf "\n"
    clear
    local next_steps=( "${NEXT_STEP_RECOVERY}" "${NEXT_STEP_RECOVERY_0}" "${NEXT_STEP_RECOVERY_1} \n\t- ${CLI_CMD} apply -f ${S3_SECRET_YAML_FILE} \n" "${NEXT_STEP_RECOVERY_2}" "${NEXT_STEP_RECOVERY_3}" "${NEXT_STEP_RECOVERY_4}" "${NEXT_STEP_RECOVERY_5}" "${NEXT_STEP_RECOVERY_6}" "${NEXT_STEP_RECOVERY_7}"  )
    display_msgs "${next_steps[@]}"

  fi 
}


function azure_blob_storage() {
  if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then
    printf "\n*************** DEBUG MSG ***********************\n"
    printf "Start of function azure_blob_storage() to retrieve user input\n"
  fi

  
  touch $AZ_SECRET_YAML_FILE
  local prompt_destination_path="Enter the destination path: "
  local prompt_storage_accout_name="Enter the storage account name:  "
  local prompt_storage_key="Enter the storage account key: "
  local prompt_storage_sas_token="Enter the storage account shared access signature (SAS): "
  local prompt_storage_connection_string="Enter the storage connection string: "
  local prompt_storage_cred_desc="In order to access your storage account, you will need one of the following combinations of credentials"
  local prompt_connection_string="1. Connection String  \n"
  local prompt_sas="2. Storage account name and Storage accout SAS Token \n"
  local prompt_stoage_key="3. Storage account name and Storage account access key \n"
  local storage_account_options=( "1" "2" "3"  )
  local storage_option="Select a valid option (1,2,3): "  

  local display=( "$prompt_connection_string" "$prompt_sas" "$prompt_stoage_key" )
  display_msgs "${display[@]}" 
  local selected_option=$( display_input_custom "$storage_option" "${storage_account_options[@]}" )
  
  
  AZURE_DESTINATION_PATH=$( enter_input  "$prompt_destination_path" )

  if [[ "${selected_option}" == "1" ]] ; then
    AZURE_ACCOUNT_STORAGE_CONN_STR=$( enter_input "$prompt_storage_connection_string" | base64 -w 0 )
    ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE data.AZURE_STORAGE_CONNECTION_STRING "${AZURE_ACCOUNT_STORAGE_CONN_STR}"
    printf "\n"
    if [[ "${MODE}" == "backup" ]] ; then

      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.connectionString.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.connectionString.key "AZURE_STORAGE_CONNECTION_STRING"
    else
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.connectionString.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.connectionString.key "AZURE_STORAGE_CONNECTION_STRING"
    fi

  elif [[ "${selected_option}" == "2" ]] ; then
    AZURE_ACCOUNT_STORAGE_NAME=$( enter_input "$prompt_storage_accout_name" | base64  -w 0 )
    # printf "\n"
    AZURE_ACCOUNT_STORAGE_SAS_TOKEN=$( enter_input "$prompt_storage_sas_token" "-s" | base64 -w 0 )
    printf "\n"

    ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE data.AZURE_STORAGE_ACCOUNT "${AZURE_ACCOUNT_STORAGE_NAME}"
    ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE data.AZURE_STORAGE_SAS_TOKEN "${AZURE_ACCOUNT_STORAGE_SAS_TOKEN}"

    if [[ "${MODE}" == "backup" ]] ; then
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageAccount.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageAccount.key "AZURE_STORAGE_ACCOUNT"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageSasToken.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageSasToken.key "AZURE_STORAGE_SAS_TOKEN"
    else 
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageAccount.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageAccount.key "AZURE_STORAGE_ACCOUNT"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageSasToken.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageSasToken.key "AZURE_STORAGE_SAS_TOKEN"
    fi

  elif [[ "${selected_option}" == "3" ]] ; then
    AZURE_ACCOUNT_STORAGE_NAME=$( enter_input "$prompt_storage_accout_name" | base64 -w 0 )
    # printf "\n"
    AZURE_ACCOUNT_STORAGE_KEY=$( enter_input "$prompt_storage_key" "-s" | base64 -w 0)
    printf "\n"
    ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE data.AZURE_STORAGE_KEY "${AZURE_ACCOUNT_STORAGE_KEY}"
    ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE data.AZURE_STORAGE_ACCOUNT "${AZURE_ACCOUNT_STORAGE_NAME}"

    if [[ "${MODE}" == "backup" ]] ; then
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageAccount.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageAccount.key "AZURE_STORAGE_ACCOUNT"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageKey.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.azureCredentials.storageKey.key "AZURE_STORAGE_KEY"
    else
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageAccount.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageAccount.key "AZURE_STORAGE_ACCOUNT"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageKey.name "${AZ_SECRET_NAME}"
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.azureCredentials.storageKey.key "AZURE_STORAGE_KEY"
      EXTERNAL_CLUSTERS=$(  ${YQ_CMD} r  $EDB_CLUSTER_YAML_FILE  -j 'spec.externalClusters' )

    fi

  fi

  if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then
    printf "\n*************** DEBUG MSG ***********************\n"
    printf "Created file ${EDB_CLUSTER_YAML_FILE} with the updated section for backup. \n"
    cat ${EDB_CLUSTER_YAML_FILE}
    printf "\n"
  fi



  ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE apiVersion "v1"
  ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE kind "Secret"
  ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE metadata.name "${AZ_SECRET_NAME}"
  ${YQ_CMD} w -i $AZ_SECRET_YAML_FILE metadata.namespace "${TARGET_PROJECT_NAME}"

  if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then
    printf "\n*************** DEBUG MSG ***********************\n"
    printf "Created file $AZ_SECRET_YAML_FILE with the updated section for backup. \n"
    cat $AZ_SECRET_YAML_FILE
    printf "\n"
  fi

  if [[ "${MODE}" == "backup" ]] ; then
    schedule_backups
    printf "\n"
    local next_steps=( "${NEXT_STEP}" "${NEXT_STEP_0}" "${NEXT_STEP_1} \n\t- ${CLI_CMD} apply -f ${AZ_SECRET_YAML_FILE} \n" "${NEXT_STEP_2}" "${NEXT_STEP_3}" )
    clear
    display_msgs "${next_steps[@]}"
  else
    printf "\n"
    local next_steps=( "${NEXT_STEP_RECOVERY}" "${NEXT_STEP_RECOVERY_0}" "${NEXT_STEP_RECOVERY_1} \n\t- ${CLI_CMD} apply -f ${AZ_SECRET_YAML_FILE} \n" "${NEXT_STEP_RECOVERY_2}" "${NEXT_STEP_RECOVERY_3}" "${NEXT_STEP_RECOVERY_4}" "${NEXT_STEP_RECOVERY_5}" "${NEXT_STEP_RECOVERY_6}" "${NEXT_STEP_RECOVERY_7}"  )
    clear
    display_msgs "${next_steps[@]}"
  fi

}


function google_cloud_storage() {

  if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then
    printf "\n*************** DEBUG MSG ***********************\n"
    printf "Start of function google_cloud_storage() to retrieve user input"
  fi



  local prompt_destination_path="Enter the destination path: "
  local prompt_google_json_file="Enter the JSON file path containing your credentials: "
  local prompt_google_auth_desc="\n You will need to get a JSON file that contains all the required information to authenticate. "
  local prompt_google_auth_desc_cont="\n Follow the instruction from Google to obtain your authentication JSON file. "
  local prompt_google_auth_desc_link="\n https://cloud.google.com/docs/authentication/client-libraries \n"
  local help_output=( "${prompt_google_auth_desc}" "${prompt_google_auth_desc_cont}" "${prompt_google_auth_desc_link}" )


  GOOGLE_DESTINATION_PATH=$( enter_input  "$prompt_destination_path" )  


  display_msgs "${help_output[@]}" 
  GOOGLE_JSON_FILE_PATH=$( enter_input "$prompt_google_json_file")
  while [[ ! -f "$GOOGLE_JSON_FILE_PATH" ]] ;
  do
    local prompt_google_json_file_empty=( "The provided file '${GOOGLE_JSON_FILE_PATH}' was not found.\n"     )
    display_msgs "${prompt_google_json_file_empty[@]}" 
    GOOGLE_JSON_FILE_PATH=$( enter_input "$prompt_google_json_file")
  done 

  if [[ -f "$GOOGLE_JSON_FILE_PATH" ]] ; then
    local prompt_google_json_file_success=( "The provided file '${GOOGLE_JSON_FILE_PATH}' was found.\n"     )
    display_msgs "${prompt_google_json_file_success[@]}"
  fi

  if [[ "${MODE}" == "backup" ]] ; then
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.destinationPath "${GOOGLE_DESTINATION_PATH}"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.googleCredentials.applicationCredentials.key "gcsCredentials"    
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.backup.barmanObjectStore.googleCredentials.applicationCredentials.name "${GOOGLE_SECRET_NAME}"
  else
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.destinationPath "${GOOGLE_DESTINATION_PATH}"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.googleCredentials.applicationCredentials.key "gcsCredentials"
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.externalClusters[0].barmanObjectStore.googleCredentials.applicationCredentials.name "${GOOGLE_SECRET_NAME}"
    EXTERNAL_CLUSTERS=$(  ${YQ_CMD} r  $EDB_CLUSTER_YAML_FILE  -j 'spec.externalClusters' )

  fi 

  # kubectl create secret generic backup-creds --from-file=gcsCredentials=gcs_credentials_file.json
  rm -fr ${GOOGLE_SECRET_SCRIPT_FILE}
  touch ${GOOGLE_SECRET_SCRIPT_FILE}
  echo -e "kubectl create secret generic backup-creds --from-file=gcsCredentials=$GOOGLE_JSON_FILE_PATH" > ${GOOGLE_SECRET_SCRIPT_FILE}

  if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then
    printf "\n*************** DEBUG MSG ***********************\n"
    printf "Created file $EDB_CLUSTER_YAML_FILE with the updated section for backup. \n"
    cat $
    printf "\n"
  fi

  if [[ "${MODE}" == "backup" ]] ; then
    schedule_backups
    printf "\n"
    local next_steps=( "${NEXT_STEP}" "${NEXT_STEP_0}" "${NEXT_STEP_1} \n\t- ${GOOGLE_SECRET_SCRIPT_FILE}\n" "${NEXT_STEP_2}" "${NEXT_STEP_3}" )
    clear
    display_msgs "${next_steps[@]}"
  else 
    printf "\n"
    local next_steps=( "${NEXT_STEP_RECOVERY}" "${NEXT_STEP_RECOVERY_0}" "${NEXT_STEP_RECOVERY_1} \n\t- ${GOOGLE_SECRET_SCRIPT_FILE}\n" "${NEXT_STEP_RECOVERY_2}" "${NEXT_STEP_RECOVERY_3}" "${NEXT_STEP_RECOVERY_4}" "${NEXT_STEP_RECOVERY_5}" "${NEXT_STEP_RECOVERY_6}" "${NEXT_STEP_RECOVERY_7}"  )
    clear
    display_msgs "${next_steps[@]}"
  fi   

}

function get_recovery_input() {
  local prompt_recovery_full_pitr_desc="How do you want to recover the database? \n"
  local prompt_recovery_full="1. In full recovery (up to the last available WAL). \n"
  local prompt_recovery_pitr="2. Up to a point in time recovery (PITR). \n"
  local mode_options=( "1" "2" )
  local prompt_valid_option="Select a valid option (1,2): "

  local display=( "$prompt_recovery_full_pitr_desc" "$prompt_recovery_full" "$prompt_recovery_pitr" )
  display_msgs "${display[@]}"
  local recovery_mode_tmp=$( display_input_custom "$prompt_valid_option" "${mode_options[@]}" )
  printf "\n"

  ## prompts for pitr input
  local prompt_pitr_desc="Select the recovery target criteria you can use: \n"
  local prompt_pitr_target_time="1. Target time (time stamp up to which recovery will proceed, expressed in RFC 3339 format) \n"
  local prompt_pitr_target_ixd="2. Target IXD (transaction ID up to which recovery will proceed. \n"
  local prompt_pitr_target_name="3. Target name (named restore point to which recovery will proceed) \n"
  local prompt_pitr_target_lsn="4. LSN of the write-ahead log location up to which recovery will proceed. \n"
  local prompt_pitr_target_immediate="5. Recovery should end as soon as a consistent state is reached (i.e. as early as possible).\n"
  local pitr_display=( "$prompt_pitr_desc" "$prompt_pitr_target_time" "$prompt_pitr_target_ixd" "$prompt_pitr_target_name" "$prompt_pitr_target_lsn" "$prompt_pitr_target_immediate" )
  local mode_pitr_options=( "1" "2" "3" "4" "5" )
  local prompt_pitr_valid_option="Select a valid option (1,2,3,4,5): "
  local prompt_pitr_backup_id_desc="Do you want to specify the base backup from which to initiate the recovery process (Yes/No Default No). "
  
  ## prompts for backupid display information
  local prompt_pitr_backup_id_important_notice=( "${RED_TEXT}While the EDB operator is able to automatically retrieve the closest backup when either targetTime or targetLSN is specified, this is not possible for the remaining targets: targetName, targetXID, and targetImmediate.${RESET_TEXT}\n" "${RED_TEXT}In such cases, it is important to specify backupID, unless you are OK with the last available backup in the catalog.${RESET_TEXT}\n") 

  case "$recovery_mode_tmp" in 
   1)
     RECOVERY_MODE="full"
     ;;
   2)
     RECOVERY_MODE="pitr"
     ;;   
  esac
 
  if [[ "${RECOVERY_MODE}" == "pitr" ]] ; then
    display_msgs "${pitr_display[@]}"   
    local pitr_mode_tmp=$( display_input_custom "$prompt_pitr_valid_option" "${mode_pitr_options[@]}" )
    case "$pitr_mode_tmp" in
      1)
        PITR_MODE="targetTime"
        local prompt_pitr_target_time_desc="Enter time stamp up to which recovery will proceed (e.g. 2020-11-26 15:22:00.00000+00 ). "
        PITR_MODE_VALUE=$( enter_input "${prompt_pitr_target_time_desc}")
        ;;
      2)
        PITR_MODE="targetIXD"
        local prompt_pitr_target_ixd_desc="Enter transaction ID up to which recovery will proceed. "
        PITR_MODE_VALUE=$( enter_input "${prompt_pitr_target_ixd_desc}")
        ;;
      3)
        PITR_MODE="targetName"
        local prompt_pitr_target_name_desc="Enter the named restore point to which recovery will proceed. "
        PITR_MODE_VALUE=$( enter_input "${prompt_pitr_target_name_desc}")
        ;;
      4)
        PITR_MODE="targetLSN"
        local prompt_pitr_target_lsn_desc="Enter LSN of the write-ahead log location up to which recovery will proceed. "
        PITR_MODE_VALUE=$( enter_input "${prompt_pitr_target_lsn_desc}")
        ;;
      5)
        PITR_MODE="targetImmediate"
        local prompt_pitr_target_immediate_desc="Recovery should end as soon as a consistent state is reached (Yes/No Default No). "
        local pitr_mode_value_tmp=$( display_input_yes_no "${prompt_pitr_target_immediate_desc}")
        if [[ "$pitr_mode_value_tmp" == "yes" ]] ; then 
           PITR_MODE_VALUE="true"
        else 
           PITR_MODE_VALUE="false"
        fi
        ;;
    esac
    printf "\n"
    # prompt_pitr_backup_id_important_notice
    display_msgs "${prompt_pitr_backup_id_important_notice[@]}"
    local pitr_enable_backup_id=$(display_input_yes_no "$prompt_pitr_backup_id_desc" )
    printf "\n"
    if [[ "$pitr_enable_backup_id" == "yes" ]] ; then
      ENABLED_BACKUP_ID="true"
      local prompt_backup_id_desc="Enter the backup id (e.g. 20220616T142236) : "
      BACKUP_ID_VALUE=$( enter_input "${prompt_backup_id_desc}")
      ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.bootstrap.recovery.recoveryTarget.backupID "${BACKUP_ID_VALUE}"
    fi
    ${YQ_CMD} w -i ${EDB_CLUSTER_YAML_FILE} spec.bootstrap.recovery.recoveryTarget.${PITR_MODE} "${PITR_MODE_VALUE}"
    
  fi
  BOOTSTRAP=$( ${YQ_CMD} r  $EDB_CLUSTER_YAML_FILE  -j 'spec.bootstrap'  )

 

}


function schedule_backups() {
  touch $EDB_CLUSTER_SCHEDULE_BACKUP

  local display_msg_schedule_backups_description="\nCreating a resource of kind type ScheduleBackup with a field called 'schedule'.\n"
  local display_msg_schedule_backups_cron_schedule="This field is a cron schedule specification, which follows the same format used in Kubernetes CronJobs.\n"
  local display_msg_schedule_backups_cron_schedule_link="https://pkg.go.dev/github.com/robfig/cron#hdr-CRON_Expression_Format\n\n"
  
  local display_msg_arr=("${display_msg_schedule_backups_description}" "${display_msg_schedule_backups_cron_schedule}" "${display_msg_schedule_backups_cron_schedule_link}"  )
  display_msgs "${display_msg_arr[@]}" 
  
  local prompt_sc_bk_cron_schedule="Enter the cron schedule specification (ex: 0 0 0 * * *): "
  SCHEDULE_BACKUPS=$( enter_input "${prompt_sc_bk_cron_schedule}" )
 

  local prompt_sc_bk_immediate="Do you want to issue a backup as soon as the ScheduledBackup resource is created? (True/False): "
  local prompt_sc_bk_immediate_options=("True", "true", "False", "false")
  local enable_sc_bk_immediate=$( display_input_custom "$prompt_sc_bk_immediate" "${prompt_sc_bk_immediate_options[@]}" )

  ${YQ_CMD} w -i $EDB_CLUSTER_SCHEDULE_BACKUP apiVersion "postgresql.k8s.enterprisedb.io/v1"
  ${YQ_CMD} w -i $EDB_CLUSTER_SCHEDULE_BACKUP kind "ScheduledBackup"
  ${YQ_CMD} w -i $EDB_CLUSTER_SCHEDULE_BACKUP metadata.name "${EDB_CLUSTER_NAME}-schedule-backup"
  ${YQ_CMD} w -i $EDB_CLUSTER_SCHEDULE_BACKUP spec.schedule "${SCHEDULE_BACKUPS}"
  # ${YQ_CMD} w -i $EDB_CLUSTER_SCHEDULE_BACKUP spec.backupOwnerReference "self"
  ${YQ_CMD} w -i $EDB_CLUSTER_SCHEDULE_BACKUP spec.immediate "${enable_sc_bk_immediate}"
  ${YQ_CMD} w -i $EDB_CLUSTER_SCHEDULE_BACKUP spec.cluster.name "${EDB_CLUSTER_NAME}"
  
  if [[ "${DEBUG}" == "True" || "${DEBUG}" == "TRUE" || "${DEBUG}" == "true" ]] ; then 
    printf "\n*************** DEBUG MSG ***********************\n"
    printf "Printing out the file $EDB_CLUSTER_SCHEDULE_BACKUP\n"
    cat "$EDB_CLUSTER_SCHEDULE_BACKUP"
  fi

  
  

}

function select_cloud_provider() {
  if [[ "$MODE" == "recovery"  ]] ; then
    local prompt_cloud_provider_description="You can recover from any service that is supported by the Barman Cloud infrastructure.\n"
  else
    local prompt_cloud_provider_description="You can archive the backup files in any service that is supported by the Barman Cloud infrastructure.\n"
  fi
  local prompt_cloud_provider_aws="1. Amazon Web Services S3  \n"
  local prompt_cloud_provider_azure="2. Microsoft Azure Blob Storage \n"
  local prompt_cloud_provider_google="3. Google Cloud Storage \n"
  local prompt_cloud_provider_compatible="You can also use any compatible implementation of the supported services.\n"
  local cloud_provider_options=( "1" "2" "3"  )
  local prompt_valid_option="Select the cloud provider that you want to archive the backup files (1,2,3): "

  local display=( "${prompt_cloud_provider_description}" "$prompt_cloud_provider_aws" "$prompt_cloud_provider_azure" "$prompt_cloud_provider_google" "$prompt_cloud_provider_compatible" )
  display_msgs "${display[@]}"
  local cloud_provider_support_tmp=$( display_input_custom "$prompt_valid_option" "${cloud_provider_options[@]}" )
  case "$cloud_provider_support_tmp" in 
   1)
     CLOUD_PROVIDER_SUPPORT="s3"
     ;;
   2)
     CLOUD_PROVIDER_SUPPORT="azure"
     ;;
   3)
     CLOUD_PROVIDER_SUPPORT="google"
     ;;
  esac 

}

function main() {

  local prompt_mode="Do you want to generate the neccessary files for backup or recovery.\n"
  local prompt_mode_backup="1. Backup \n"
  local prompt_mode_recovery="2. Recovery \n"
  local mode_options=( "1" "2" )
  local prompt_valid_option="Select a valid option (1,2): "

  local display=( "$prompt_mode" "$prompt_mode_backup" "$prompt_mode_recovery" )
  display_msgs "${display[@]}"
  local mode=$( display_input_custom "$prompt_valid_option" "${mode_options[@]}" )
  case "$mode" in
  1)
    MODE="backup"
    ;;
  2)
    MODE="recovery"
    ;;
  esac

  printf "\n"
  select_cloud_provider
  select_project
  get_edb_cluster
  if [[ "$MODE" == "recovery" ]] ; then
    get_recovery_input
  fi

  case "$CLOUD_PROVIDER_SUPPORT" in
  aws|s3)
    aws_s3
    ;;
  azure)
    azure_blob_storage
    ;;
  google)
    google_cloud_storage
    ;; 
  esac
  
}

main
exit 0
